#!/usr/bin/env python3
#    wlchat, a chat client for WhiteLeaf's fork of MemeLabs.
#    Copyright (C) 2022-2025  Alicia <alicia@ion.nu>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GdkPixbuf
from gi.repository import GLib
from gi.repository import Gio
import json
import wlchat
import base64
import time

def gprint_(c, text, tag=None):
  i=c['chat'].get_end_iter()
  if(tag):
    c['chat'].insert_with_tags_by_name(i, text, (tag))
  else:
    c['chat'].insert(i, text)
  adj=c['scroll'].get_vadjustment()
  if(adj.get_value()+adj.get_page_size()+50>=adj.get_upper()):
    GLib.idle_add(scrollchat, 0, 0, c)
  lines=c['chat'].get_line_count()
  if(lines>2048): # Limit scrollback. TODO: Make 2048 lines configurable
    start=c['chat'].get_start_iter()
    end=c['chat'].get_iter_at_line(lines-2048)
    c['chat'].delete(start, end)

def gprint(c, text, tag=None):
  GLib.idle_add(lambda:gprint_(c, text, tag))

def insertimg(c, imgfile, tooltip):
  img=Gtk.Image.new_from_file(imgfile)
  img.set_tooltip_text(tooltip)
  anchor=c['chat'].create_child_anchor(c['chat'].get_end_iter())
  c['chattext'].add_child_at_anchor(img, anchor)
  img.show()
  c['emotefreeze'].append([anchor, img]) # Add to the queue to be frozen when scrolled out of view. TODO: Make this optional

def insertpronoun(c, pronoun):
  pronoun=wlchat.pronouns[int(pronoun)].split('/')
  tag=pronoun[0]+pronoun[1] # Tag for pronoun color
  pronoun=pronoun[0][0]+'/'+pronoun[1][0]
  gprint(c, ' '+pronoun, tag)

def msg(c, chan, nick, data):
  if(data['data'].startswith('/me ')): # According to cripton86 webclient doesn't do greentext for /me
    gprint(c, '\n* ')
    gprint(c, data['nick'], 'name')
    if(data.get('pronouns') and int(data['pronouns'])>0 and int(data['pronouns'])<len(wlchat.pronouns)): insertpronoun(c, data['pronouns'])
    data['data']=data['data'][3:]
    wlchat.findemotes(data, c['emotes']) # Do it here so we get the right positions
  else:
    wlchat.findemotes(data, c['emotes'])
    gprint(c, '\n'+data['nick'], 'name')
    if(data.get('pronouns') and int(data['pronouns'])>0 and int(data['pronouns'])<len(wlchat.pronouns)): insertpronoun(c, data['pronouns'])
    gprint(c, ': ')
  if(data.get('nodes') and data['nodes'].get('emotes')):
    last=0
    green=None
    for emote in data['nodes']['emotes']:
      start=emote['bounds'][0]
      end=emote['bounds'][1]
      if(not green and data['data'][last:start].count('>')>0):
        green='greentext'
        gt=data['data'][last:start].index('>')
        gprint(c, data['data'][last:last+gt])
        gprint(c, data['data'][last+gt:start], green)
      else:
        gprint(c, data['data'][last:start], green)
      imgfile=wlchat.get_emote_img(emote, c['emotes'], chan)
      tooltip=emote['name']
      if(emote.get('modifiers')):
        tooltip+=':'+(':'.join(emote['modifiers']))
      GLib.idle_add(insertimg, c, imgfile, tooltip)
      last=end
    data['data']=data['data'][last:]
  if(data['data'].count('>')>0): # Greentext
    gt=data['data'].index('>')
    gprint(c, data['data'][:gt])
    gprint(c, data['data'][gt:], 'greentext')
  else:
    gprint(c, data['data'])

def privmsg(c, chan, nick, data):
  gprint(c, '\nPrivate message from '+data['nick']+': '+data['data'])

def broadcast(c, chan, nick, data):
  gprint(c, '\nBroadcast: '+data['data'])
  if(data.get('data')=='emoteupdate'): # Try to update emotes without a restart
    c['emotes']=wlchat.fetch_emotes(c['site'])

def namelist_del(nl, user):
  i=0
  x=nl.get_row_at_index(i)
  while(x):
    txt=x.get_child()
    if(txt.get_text()==user['nick']):
      txt.destroy()
      nl.remove(x)
      x.destroy()
    else:
      i+=1
    x=nl.get_row_at_index(i)

def namelist_add(nl, user):
  namelist_del(nl, user) # Prevent duplicates
  label=Gtk.Label.new(user['nick'])
  label.set_halign(Gtk.Align.START)
  nl.add(label)
  label.show()

def namelist(c, chan, nick, data):
  for user in data:
    GLib.idle_add(namelist_add, c['users'], user)

def join(c, chan, nick, data):
  gprint(c, '\nJoin: '+data['nick'], 'join')
  if(data.get('pronouns') and int(data['pronouns'])>0 and int(data['pronouns'])<len(wlchat.pronouns)): insertpronoun(c, data['pronouns'])
  GLib.idle_add(namelist_add, c['users'], data)

def quitmsg(c, chan, nick, data):
  gprint(c, '\nQuit: '+data['nick'], 'quit')
  GLib.idle_add(namelist_del, c['users'], data)

def notice(c, chan, nick, data):
  gprint(c, '\nNotice: '+data)

def register(c, chan, formdata, captcha):
  window=Gtk.ApplicationWindow(application=app)
  window.set_title('WLChat Register ('+chan+')')
  grid=Gtk.Grid()
  # Username
  grid.attach(Gtk.Label.new('Username:'), 0,0,1,1)
  username=Gtk.Entry.new()
  grid.attach(username, 1,0,1,1)
  # Agreement
  agreement=Gtk.CheckButton.new_with_label('Accept agreement')
  grid.attach(agreement, 0,1,1,1)
  link=Gtk.Label.new()
  link.set_markup('<a href="https://'+chan+'/agreement">Agreement</a>')
  grid.attach(link, 1,1,2,1)
  # Captcha
  captchabin=base64.decodebytes(bytes(captcha.split(',')[1], 'utf8'))
  loader=GdkPixbuf.PixbufLoader.new()
  loader.write(captchabin)
  loader.close()
  img=loader.get_pixbuf()
  img=Gtk.Image.new_from_pixbuf(img)
  grid.attach(img, 0,2,1,1)
  captcha=Gtk.Entry.new()
  grid.attach(captcha, 1,2,1,1)
  # Button
  btn=Gtk.Button.new_with_label('Register')
  def doregister(x,w, username, captcha, agreement, formdata):
    formdata['username']=username.get_text()
    formdata['catch']=captcha.get_text()
    if(agreement.get_active()):
      formdata['agreement']='on'
    w.destroy()
  btn.connect('clicked', doregister, window, username, captcha, agreement, formdata)
  grid.attach(btn, 0,3,2,1)
  window.add(grid)
  window.show_all()
  while(window.get_visible()):
    time.sleep(0.1)
    while(Gtk.events_pending()): Gtk.main_iteration()

interface={
  'msg': msg,
  'privmsg': privmsg,
  'broadcast': broadcast,
  'namelist': namelist,
  'join': join,
  'quit': quitmsg,
  'notice': notice,
  'register': register,
}

def scrollchat(x, y, c):
  adj=c['scroll'].get_vadjustment()
  adj.set_value(adj.get_upper()+1000)
  while(len(c['emotefreeze'])>0): # Freeze emotes scrolled out of view
    if(not c['emotefreeze'][0][0].get_deleted()):
      itr=c['chat'].get_iter_at_child_anchor(c['emotefreeze'][0][0])
      rect=c['chattext'].get_iter_location(itr)
      x,y=c['chattext'].buffer_to_window_coords(Gtk.TextWindowType.TEXT, rect.x, rect.y)
      if(y>-200): break # Not far enough out of view
      pb=c['emotefreeze'][0][1].get_animation()
      if(pb):
        pb=pb.get_static_image()
        c['emotefreeze'][0][1].set_from_pixbuf(pb)
    c['emotefreeze'].remove(c['emotefreeze'][0])

def sortnames(row1, row2):
  a=row1.get_child().get_text()
  b=row2.get_child().get_text()
  if(a<b): return -1
  if(a>b): return 1
  return 0

def sendmsg(entry, c):
  txt=entry.get_text()
  if(txt=='/emoteupdate'):
    c['emotes']=wlchat.fetch_emotes(c['site'])
  else:
    c['conn'][0].send('MSG '+json.dumps({'data':txt}))
  entry.set_text('')

def searchemote(entry, emotes):
  q=entry.get_text().lower()
  for e in emotes:
    if(emotes[e].get('prefix')):
      if(emotes[e]['prefix'].lower().count(q)>0):
        emotes[e]['widget'].show()
      else:
        emotes[e]['widget'].hide()

def insertemote(x, name, inputfield, search):
  search.set_text('')
  inputfield.set_text(inputfield.get_text()+name)
  inputfield.set_position(len(inputfield.get_text()))

def inputkeyevent(field, event, c):
  if(event.keyval==Gdk.KEY_Tab or event.keyval==Gdk.KEY_KP_Tab):
    pos=field.get_position()
    text=field.get_text()
    start=pos
    while(start>0 and text[start-1:start]!=' '):
      start-=1
    query=text[start:pos]
    casesensitive=(query!=query.lower())
    match=False
    fullmatch=False
    # Look for matching users
    i=0
    x=c['users'].get_row_at_index(i)
    while(x):
      txt=x.get_child()
      txt=txt.get_text()
      txtorig=txt
      if(not casesensitive): txt=txt.lower()
      if(txt.startswith(query)):
        if(match==False):
          match=txt
        else:
          j=1
          while(j<len(match)):
            if(txt[:j]!=match[:j]):
              match=match[:j-1]
            j+=1
        if(match==txt): fullmatch=txtorig
      i+=1
      x=c['users'].get_row_at_index(i)
    # Look for matching emotes
    for txt in c['emotes']:
      txtorig=txt
      if(not casesensitive): txt=txt.lower()
      if(txt.startswith(query)):
        if(match==False):
          match=txt
        else:
          j=1
          while(j<len(match)):
            if(txt[:j]!=match[:j]):
              match=match[:j-1]
            j+=1
        if(match==txt): fullmatch=txtorig
    if(not match): return True
    if(fullmatch and not casesensitive and fullmatch.lower()==match.lower()): match=fullmatch+' '
    elif(fullmatch and casesensitive and fullmatch==match): match=fullmatch+' '
    # Insert completion
    text=text[:start]+match+text[pos:]
    field.set_text(text)
    field.set_position(start+len(match))
    return True
  return False

def app_join(button, siteentry, browser, browserprofile, authmethod):
  site=siteentry.get_text()
  siteentry.set_text('')
  browser=browser.get_active_text()
  browserprofile=browserprofile.get_filename()
  authmethod=authmethod.get_active_text().lower()
  textview=Gtk.TextView()
  textview.set_editable(False)
  textview.set_wrap_mode(Gtk.WrapMode.WORD_CHAR)
  textview.set_can_focus(False)
  textstyle=textview.get_style_context()
  css=Gtk.CssProvider()
  css.load_from_data('* {background-color:#202020;color:#ffffff;}');
  textstyle.add_provider(css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
  chatscroll=Gtk.ScrolledWindow.new(None, None)
  chatscroll.add(textview)
  chatscroll.set_hexpand(True)
  chatscroll.set_vexpand(True)
  userlist=Gtk.ListBox()
  userlist.set_sort_func(sortnames)
  userscroll=Gtk.ScrolledWindow.new(None, None)
  userscroll.add(userlist)
  inputfield=Gtk.Entry()
  inputfield.add_events(Gdk.EventMask.KEY_PRESS_MASK)
  emotebutton=Gtk.MenuButton.new()
  emotebutton.add(Gtk.Label.new('Emotes'))
  emotemenu=Gtk.Menu()
  emotebutton.set_popup(emotemenu)
  emotesearch=Gtk.Entry()
  emotesearchitem=Gtk.MenuItem.new()
  emotesearchitem.add(emotesearch)
  emotemenu.attach(emotesearchitem, 0, 10, 0, 1)
  emotebutton.connect('clicked', lambda x, search: search.grab_focus(), emotesearch)
  grid=Gtk.Grid()
  grid.attach(chatscroll, 0,0,1,1)
  grid.attach(userscroll, 1,0,2,1)
  grid.attach(inputfield, 0,1,2,1)
  grid.attach(emotebutton, 2,1,1,1)
  tab=Gtk.Box.new(Gtk.Orientation.HORIZONTAL, 0)
  tab.pack_start(Gtk.Label.new(site.replace('www.','')), False, False, 0)
  # Add a close button
  closebtn=Gtk.Button.new_from_icon_name('window-close', Gtk.IconSize.LARGE_TOOLBAR)
  tab.pack_start(closebtn, False, False, 0)
  tab.show_all()
  pagenum=chatlist.append_page(grid, tab)
  grid.show_all()
  chatlist.set_current_page(pagenum)
  conn=[None]
  chatbuf=textview.get_buffer()
  tags=chatbuf.get_tag_table()
  tag=Gtk.TextTag.new('join')
  tag.set_property('foreground', '#0080ff')
  tags.add(tag)
  tag=Gtk.TextTag.new('quit')
  tag.set_property('foreground', '#000080')
  tags.add(tag)
  tag=Gtk.TextTag.new('greentext')
  tag.set_property('foreground', '#00a000')
  tags.add(tag)
  tag=Gtk.TextTag.new('name')
  tag.set_property('weight', 800)
  tags.add(tag)
  # Pronoun color tags, colors found from grep -o 'pronoun\.[a-z]*:after{content:"[a-zA-Z/]*";color:#[0-9a-f]*;background-color:#[0-9a-f]*}' /tmp/chat.e7d595.css | sed -e 's/.*content:"//; s/".*background-color:/ = /; s/}//'
  for pn in [('Any/All','#ff8686'),('She/Her','#f286ff'),('He/Him','#86b6ff'),('They/Them','#86ff8f'),('It/Its','#fffb86'),('He/They','#86fff9'),('She/They','#ca78ff'),('Ask/Me','#bcbcbc'),('Name/They','#d8ffa5')]:
    tag=Gtk.TextTag.new(pn[0].replace('/','').lower())
    tag.set_property('foreground', pn[1])
    tags.add(tag)
  # Chat-specific context
  c={'chat':chatbuf,'chattext':textview,'users':userlist,'scroll':chatscroll,'emotes':wlchat.fetch_emotes(site),'site':site,'conn':conn,'emotefreeze':[]}
  emotesearch.connect('changed', searchemote, c['emotes'])
  i=0
  for e in c['emotes']:
    emote=Gtk.MenuItem.new()
    img=wlchat.get_emote_img({'name':c['emotes'][e]['prefix']}, c['emotes'], site)
    # Sometimes int, sometimes string with additional junk, so we end up converting to string first
    if(int(str(c['emotes'][e]['imageWidth']).strip('px'))>32):
      try:
        pb=GdkPixbuf.Pixbuf.new_from_file(img)
        pb=GdkPixbuf.Pixbuf.new_subpixbuf(pb, (pb.get_width()-32)/2, 0, 32, 32)
        img=Gtk.Image.new_from_pixbuf(pb)
      except:
        print('Fell back on fullsize for '+img+' ('+c['emotes'][e]['prefix']+')')
        img=Gtk.Image.new_from_file(img)
    else:
      img=Gtk.Image.new_from_file(img)
    img.set_tooltip_text(c['emotes'][e]['prefix'])
    emote.add(img)
    emote.connect('activate', insertemote, c['emotes'][e]['prefix'], inputfield, emotesearch)
    c['emotes'][e]['widget']=emote # For search
    emotemenu.attach(emote, i%10, (i%10)+1, int(i/10)+1, int(i/10)+2)
    i+=1
  emotemenu.show_all()
  adj=chatscroll.get_vadjustment()
  inputfield.connect('activate', sendmsg, c)
  inputfield.connect('key-press-event', inputkeyevent, c)
  def closetab(page,c):
    chatlist.remove_page(page)
    c['conn'][0].on_close=None
    c['conn'][0].on_error=None
    c['conn'][0].on_message=None
    c['conn'][0].close()
  closebtn.connect('clicked', lambda x,page,c: closetab(page,c), pagenum, c)
  wlchat.joinchat(site, browser, browserprofile, '', conn, interface, c, authmethod)

def app_jointab(app):
  auth=Gtk.ComboBoxText()
  for x in wlchat.authmethods:
    auth.insert(-1, None, x[0].upper()+x[1:])
  auth.set_active(0)
  site=Gtk.Entry()
  browser=Gtk.ComboBoxText()
  for b in wlchat.cookies.validbrowsers:
    browser.insert(-1, None, b)
  browserprofile=Gtk.FileChooserButton(title='Select browser profile', action=Gtk.FileChooserAction.SELECT_FOLDER)
  button=Gtk.Button.new_with_label('Join chat')
  button.connect('clicked', app_join, site, browser, browserprofile, auth)
  grid=Gtk.Grid()
  label=Gtk.Label.new('Website: ')
  label.set_halign(Gtk.Align.END)
  grid.attach(label, 0,0,1,1)
  grid.attach(site, 1,0,1,1)
  label=Gtk.Label.new('Authentication method: ')
  label.set_halign(Gtk.Align.END)
  grid.attach(label, 0,1,1,1)
  grid.attach(auth, 1,1,1,1)
  label=Gtk.Label.new('Browser to grab login: ')
  label.set_halign(Gtk.Align.END)
  grid.attach(label, 0,2,1,1)
  grid.attach(browser, 1,2,1,1)
  label=Gtk.Label.new('Browser profile: ')
  label.set_halign(Gtk.Align.END)
  grid.attach(label, 0,3,1,1)
  grid.attach(browserprofile, 1,3,1,1)
  grid.attach(button, 0,4,2,1)
  chatlist.append_page(grid, Gtk.Image.new_from_icon_name('list-add', Gtk.IconSize.LARGE_TOOLBAR))
  grid.show_all()
  site.grab_focus()

def app_init(app):
  global chatlist
  global chatwindow
  chatwindow=Gtk.ApplicationWindow(application=app)
  chatwindow.set_title('WLChat')
  chatlist=Gtk.Notebook()
  chatwindow.add(chatlist)
  chatwindow.connect('delete-event', lambda x, y: app.quit())
  app_jointab(app)
  chatwindow.show_all()

app=Gtk.Application(application_id='org.ion.wlchat', flags=Gio.ApplicationFlags.NON_UNIQUE)
app.connect('activate', app_init)

app.run(None)
