#!/usr/bin/env python3
#    wlchat, a chat client for WhiteLeaf's fork of MemeLabs.
#    Copyright (C) 2022-2025  Alicia <alicia@ion.nu>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

import wlchat
import socket
import threading
import sys
import json
port=3333
pcolors=[ # Pronoun color associations
  '00', # 1-indexed, so pad 0
  '12',
  '13',
  '06',
  '02',
  '04',
  '15',
  '00',
  '08',
  '06'
]

def irc_msg(c, chan, nick, data):
  if(data['nick']==nick): # IRC client prints our messages, don't re-send them
    return
  data['data']=data['data'].replace('\n', ' ')
  # Process emotes
  if(data.get('nodes') and data['nodes'].get('emotes')):
    emoteoffset=0
    for emote in data['nodes']['emotes']:
      start=emote['bounds'][0]+emoteoffset
      end=emote['bounds'][1]+emoteoffset
      data['data']=data['data'][0:start]+'[\x02'+data['data'][start:end]+'\x02]'+data['data'][end:]
      emoteoffset+=4
  if(data.get("pronouns") and int(data['pronouns'])>0 and int(data['pronouns'])<len(wlchat.pronouns)):
    pos=0
    if(data['data'].startswith('/me ')):
      pos+=4
    # Instead of full pronouns, just add initials and color to the beginning of the message
    pronoun=wlchat.pronouns[int(data['pronouns'])].split('/')
    pronoun=pronoun[0][0]+'/'+pronoun[1][0]
    data['data']=data['data'][0:pos]+'\x03'+pcolors[int(data['pronouns'])]+pronoun+'\x0f '+data['data'][pos:]
  # Translate /me
  if(data['data'].startswith('/me ')):
    data['data']='\x01ACTION '+data['data'][4:]+'\x01'
  # Send to IRC
  c['irc'].send(bytes(':'+data['nick']+'!user@host PRIVMSG #'+chan+' :'+data['data']+'\n', 'utf-8'))

def irc_privmsg(c, chan, nick, data):
  c['irc'].send(bytes(':'+chan+'|'+data['nick']+'!user@host PRIVMSG '+nick+' :'+data['data']+'\n', 'utf-8'))

def irc_broadcast(c, chan, nick, data):
  c['irc'].send(bytes(':wlchat NOTICE #'+chan+' :Broadcast: '+data['data']+'\n', 'utf-8'))

def irc_namelist(c, chan, nick, userlist):
  ulist=''
  for user in userlist:
    if(ulist!=''):
      ulist+=' '
    if(user.get('features') and user['features'].count('moderator')>0):
      ulist+='@'+user['nick']
    else:
      ulist+=user['nick']
  c['irc'].send(bytes(':wlchat 353 '+nick+' = #'+chan+' :'+ulist+'\n', 'utf-8'))
  c['irc'].send(bytes(':wlchat 366 '+nick+' #'+chan+' :End of /NAMES list.\n', 'utf-8'))

def irc_join(c, chan, nick, data):
  if(data['nick']==nick):
    if(not c['firstjoin']): return
    c['firstjoin']=False
  c['irc'].send(bytes(':'+data['nick']+'!user@host JOIN #'+chan+'\n', 'utf-8'))
  if(data.get('features') and data['features'].count('moderator')>0):
    c['irc'].send(bytes(':wlchat MODE #'+chan+' +o '+data['nick']+'\n', 'utf-8'))

def irc_quit(c, chan, nick, data):
  if(data['nick']==nick): return
  c['irc'].send(bytes(':'+data['nick']+'!user@host PART #'+chan+' :\n', 'utf-8'))

def irc_notice(c, chan, nick, data):
  c['irc'].send(bytes(':wlchat NOTICE #'+chan+' :'+data+'\n', 'utf-8'))

def irc_register(c, chan, formdata, captcha):
  c['irc'].send(bytes(':wlchat NOTICE #'+chan+' :Captcha: '+captcha+' (reply to answer, and implicitly agree to the user agreement at https://'+chan+'/agreement)\n', 'utf-8'))
  while(True):
    line=c['irc'].recv(1024)
    if(line.startswith(b'PRIVMSG ')):
      chanmsg=bytes.decode(line.split(b' ')[1])
      if(chanmsg.startswith('#')):
        chanmsg=chanmsg[1:]
      if(chanmsg!=chan):
        c['irc'].send(bytes(':wlchat NOTICE #'+chanmsg+' :Failed to send message. Awaiting captcha response for #'+chan+'\n', 'utf-8'))
      else:
        msg=bytes.decode(line[line.find(b' ', 8)+1:])
        if(msg.startswith(':')):
          msg=msg[1:]
        formdata['catch']=msg.replace('\n','').replace('\r','')
        formdata['agreement']='on'
        break
    elif(line.startswith(b'PING ')):
      conn.send(b':wlchat PONG '+line[5:]+b'\n')

def irc_topic(c, chan, nick, data):
  c['irc'].send(bytes(':wlchat TOPIC #'+chan+' :'+data+'\n', 'utf-8'))

def irc_mainloop(conn):
# TODO: Also keep a timestamp of the latest ping received from wl (but local time, not ping's), reconnect if 30s old
  browser=''
  browserprofile=''
  nick=''
  ctx=None
  while(True):
    try:
      line=conn.recv(1024)
    except Exception: break
    if(line==b''): # EOF
      break
    for line in line.split(b'\n'):
      if(line.startswith(b'CAP ')):
        conn.send(b'CAP * LS :\n')
      elif(line.startswith(b'PRIVMSG ')):
        chan=bytes.decode(line.split(b' ')[1])
        if(ctx!=None):
          msg=bytes.decode(line[line.find(b' ', 8)+1:]).strip('\r\n')
          if(msg.startswith(':')):
            msg=msg[1:]
          # Translate /me
          if(msg.startswith('\x01ACTION ')):
            msg='/me '+msg[8:]
            if(msg.endswith('\x01')):
              msg=msg[0:-1]
          # Send to wl
          try:
            if(chan.startswith('#')):
              ctx['wl'][0].send('MSG '+json.dumps({'data':msg}))
            else:
              ctx['wl'][0].send('PRIVMSG '+json.dumps({'nick':chan,'data':msg}))
          except Exception as e:
            x=sys.exc_info()
            conn.send(bytes(':wlchat NOTICE '+chan+' :Exception(2): "'+str(e)+'" Line '+str(x[2].tb_lineno)+'\n', 'utf-8'))
            sys.stdout.flush()
      elif(line.startswith(b'PING ')):
        conn.send(b':wlchat PONG '+line[5:]+b'\n')
      elif(line.startswith(b'USER ')):
        browser=bytes.decode(line.split(b' ')[1])
      elif(line.startswith(b'PASS ')):
        browserprofile=bytes.decode(line[5:].strip())
        if(browserprofile.startswith(':')):
          browserprofile=browserprofile[1:]
      elif(line.startswith(b'NICK ')):
        newnick=line[5:].strip()
        if(newnick.startswith(b':')):
          newnick=newnick[1:]
        if(nick==''):
          conn.send(b':wlchat 001 '+newnick+b' :Welcome\n')
        else:
          conn.send(b':'+bytes(nick, 'utf-8')+b'!user@host NICK :'+newnick+b'\n')
        nick=bytes.decode(newnick)
      elif(line.startswith(b'JOIN ')):
        chan=bytes.decode(line[5:].strip())
        if(chan.startswith(':')):
          chan=chan[1:]
        if(chan.startswith('#')):
          chan=chan[1:]
        if(chan==''): # Ignore weird empty joins
          continue
        if(ctx!=None): # Send ERR_TOOMANYCHANNELS
          conn.send(b':wlchat 405 #'+chan.encode('UTF-8')+b' :Can\'t join more than one site per connect if we want to keep track of DMs\n')
          continue
        ctx={'irc':conn, 'channel':chan, 'wl':[None], 'firstjoin':True}
        wlchat.joinchat(chan, browser, browserprofile, nick, ctx['wl'], interface, ctx)
        conn.send(b':'+nick.encode('UTF-8')+b'!user@host JOIN #'+chan.encode('UTF-8')+b'\n')
      elif(line.startswith(b'PART ')):
        chan=bytes.decode(line[5:].strip())
        if(chan.startswith(':')):
          chan=chan[1:]
        if(chan.startswith('#')):
          chan=chan[1:]
        if(chan==''): # Ignore weird empty parts (inherited from JOIN, not sure empty parts happen)
          continue
        ctx[0].close()
        ctx[0]=None
  print('Disconnected from IRC. Closing wl connections...')
  ctx['wl'][0].on_close=None
  ctx['wl'][0].on_error=None
  ctx['wl'][0].on_message=None
  ctx['wl'][0].close()

interface={
  'msg': irc_msg,
  'privmsg': irc_privmsg,
  'broadcast': irc_broadcast,
  'namelist': irc_namelist,
  'join': irc_join,
  'quit': irc_quit,
  'notice': irc_notice,
  'register': irc_register,
  'topic': irc_topic,
}

# Set up server-based interface
s=socket.socket(socket.AF_INET, socket.SOCK_STREAM)
s.bind(('127.0.0.1', port))
s.listen()
print('Listening to localhost:'+str(port))
while(True):
  conn, addr=s.accept()
  print('Got connection')
  thread=threading.Thread(target=irc_mainloop, args=[conn])
  thread.daemon=True
  thread.start()
