#!/usr/bin/env python3
#    wlchat, a chat client for WhiteLeaf's fork of MemeLabs.
#    Copyright (C) 2022-2025  Alicia <alicia@ion.nu>
#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU Affero General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU Affero General Public License for more details.
#
#    You should have received a copy of the GNU Affero General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

import wlchat
import sys
import os
import requests
import json
import signal
if(os.environ.get('TERMINOLOGY')):
  import PIL.ImageFile

def terminal_print(txt):
  lines=os.get_terminal_size().lines
  sys.stdout.write('\x1b[s')
  sys.stdout.write('\x1b[0;'+str(lines-2)+'r')
  sys.stdout.write('\x1b['+str(lines-2)+';0H')
  sys.stdout.write('\n'+txt)
  sys.stdout.write('\x1b['+str(lines)+';'+str(lines)+'r')
  sys.stdout.write('\x1b['+str(lines)+';0H> ')
  sys.stdout.write('\x1b[u')
  sys.stdout.flush()

def terminal_msg(emotes, chan, nick, data):
  wlchat.findemotes(data, emotes)
  # Download and cache emotes
  if(data.get('nodes') and data['nodes'].get('emotes')):
    emoteoffset=0
    for emote in data['nodes']['emotes']:
      start=emote['bounds'][0]+emoteoffset
      end=emote['bounds'][1]+emoteoffset
      if(os.environ.get('TERMINOLOGY')):
        path=wlchat.get_emote_img(emote, emotes, chan)
        img=PIL.Image.open(path)
        width=round(img.size[0]/16)
        emoteimg='\x1b}is#'+str(width)+';1;'+path+'\x00\x1b}ib\x00'+('#'*width)+'\x1b}ie\x00'
      else:
        emoteimg='[\x1b[1m'+emote['name']+'\x1b[0m]'
      data['data']=data['data'][0:start]+emoteimg+data['data'][end:]
      emoteoffset+=len(emoteimg)-(end-start)
  pronoun=''
  if(data.get("pronouns") and int(data['pronouns'])>0 and int(data['pronouns'])<len(wlchat.pronouns)):
    pronoun=wlchat.pronouns[int(data['pronouns'])]
  if(data['data'].startswith('/me ')):
    data['data']=data['data'][4:]
    if(pronoun!=''): pronoun=' ('+pronoun+')'
    terminal_print(' \x1b[1m'+data['nick']+'\x1b[0m'+pronoun+' '+data['data'])
  else:
    if(pronoun!=''): pronoun=' '+pronoun
    terminal_print('<\x1b[1m'+data['nick']+'\x1b[0m'+pronoun+'> '+data['data'])

def terminal_privmsg(c, chan, nick, data):
  True

def terminal_broadcast(c, chan, nick, data):
  True

def terminal_namelist(c, chan, nick, userlist):
  ulist=''
  for user in userlist:
    if(ulist!=''):
      ulist+=', '
    ulist+=user['nick']
  terminal_print('Users online:')
  terminal_print(ulist)

def terminal_join(c, chan, nick, data):
  terminal_print('Join: '+data['nick'])

def terminal_quit(c, chan, nick, data):
  terminal_print('Quit: '+data['nick'])

def terminal_notice(c, chan, nick, data):
  terminal_print('Notice: '+data)

def terminal_register(c, chan, formdata, captcha):
  print('New account')
  print('Site username:')
  formdata['username']=sys.stdin.readline().replace('\n', '')
  print(captcha)
  print('Captcha:')
  formdata['catch']=sys.stdin.readline().replace('\n', '')
  print('Accept user agreement? (https://'+chan+'/agreement)')
  if(sys.stdin.readline()[0].lower()=='y'):
    formdata['agreement']='on'

def terminal_mainloop():
  print('Chat to join: (full domain)')
  domain=sys.stdin.readline().replace('\n', '')
  browser='_'
  browserprofile=''
  while(browser!='' and wlchat.cookies.validbrowsers.count(browser)==0):
    print('Browser (one of '+(', '.join(wlchat.cookies.validbrowsers))+', or empty for no login):')
    browser=sys.stdin.readline().replace('\n', '')
  if(browser!=''):
    print('Browser profile (or empty for default):')
    browserprofile=sys.stdin.readline().replace('\n', '')
  nick=''
  conn=[None]
  # Gather emote data
  emotelist=wlchat.fetch_emotes(domain)
  lines=os.get_terminal_size().lines
  wlchat.joinchat(domain, browser, browserprofile, nick, conn, interface, emotelist)
  sys.stdout.write('\x1b[2J\x1b['+str(lines)+';3H') # Clear screen
  sys.stdout.write('\x1b['+str(lines)+';0H> ') # Input prompt
  msg=sys.stdin.readline()
  while(msg!=''):
    sys.stdout.write('\n> ')
    sys.stdout.flush()
    msg=msg.replace('\n','')
    if(msg!=''):
      # Send to wl
      conn[0].send('MSG '+json.dumps({'data':msg}))
    msg=sys.stdin.readline()
  print('\x1b[r\n')

interface={
  'msg': terminal_msg,
  'privmsg': terminal_privmsg,
  'broadcast': terminal_broadcast,
  'namelist': terminal_namelist,
  'join': terminal_join,
  'quit': terminal_quit,
  'notice': terminal_notice,
  'register': terminal_register,
}

def handle_sigint(a,b):
  sys.stdout.write('\n\n> ')

signal.signal(signal.SIGINT, handle_sigint)
terminal_mainloop()
